//	TorusGamesGraphicsViewMac.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesGraphicsViewMac.h"
#import "TorusGames-Common.h"
#import "TorusGamesRenderer.h"
#import "GeometryGamesModel.h"
#import "GeometryGamesUtilities-Common.h"
#import "GeometryGamesUtilities-Mac-iOS.h"
#import "GeometryGamesUtilities-Mac.h"
#import "GeometryGamesLocalization.h"	//	for IsCurrentLanguage()
#import <QuartzCore/QuartzCore.h>		//	for CAMetalLayer


//	Trackpad behavior
//
//	If the user has enabled "Tap to click" in the system preferences,
//	then a tap-and-drag on the trackpad will generate a MouseDown.
//	If, while such a drag is in progress, the user lowers a second finger,
//	continues dragging, lifts the second figure and finally lifts
//	the first finger, the sequence of events will be
//
//		MouseDown()		//	for single finger
//		MouseDown()		//	for 2-finger gesture
//		MouseUp()		//	for 2-finger gesture
//		MouseUp()		//	for single finger
//
//	The nested mouse events are a little disturbing,
//	even though they do no damage.
//
//	Even without "Tap to click" enabled, the user may freely
//	mix clicks on a physical mouse with 2-finger trackpad gestures.
//	So we need to be prepared for a real mess!
//
//	Unlike the purely mouse-related problem of mouseDragged calls
//	arriving with no previous mouseDown call, which was clearly
//	Apple's bug (in the sense that an NSWindow doesn't always
//	dispatch NSLeftMouseDragged events to the same NSView that
//	got the initial NSLeftMouseDown event) (BUT HAS NOW BEEN FIXED,
//	AS OF MACOS 10.12.4 AND MAYBE EARLIER), this trackpad issue
//	is purely my own fault, for letting my implementations
//	of mouseDown and touchesBeganWithEvent both call
//	the same MouseDown() function.  On the other hand,
//	that really is the desired functionality -- in effect
//	it's like having two mice connected to the same cursor.
//	(2017-04-07 This problem has now been ameliorated
//	by using aTwoFingerGestureFlag to request two-finger scrolls.
//	Using that flag separates two-finger gesture processing
//	from regular mouse event processing.)


//	Privately-declared methods
@interface TorusGamesGraphicsViewMac()
- (void)characterInput:(unichar)aCharacter;
- (void)mouseDown:(NSEvent *)anEvent withRightButton:(bool)aRightClick;
- (CGPoint)gestureLocation:(NSSet *)aTouchSet;
@end


@implementation TorusGamesGraphicsViewMac
{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	//	Does the cursor live on the torus or Klein bottle?
	bool	itsTorusCursorFlag;		//	used for 2D games only
#endif

	//	Interpret a two-finger trackpad gesture as a scroll.
	bool	itsTwoFingerGestureIsInProgress;
	CGPoint	itsPrevGestureLocation;	//	in [-0.5, +0.5] trackpad coordinates
}

- (id)initWithModel:(GeometryGamesModel *)aModel frame:(NSRect)aFrame
{
	self = [super initWithModel:aModel frame:aFrame];
	if (self != nil)
	{
		CALayer	*theLayer;

		//	Create itsRenderer.

		//	The superclass has already installed a CAMetalLayer.
		theLayer = [self layer];

		GEOMETRY_GAMES_ASSERT(
			[theLayer isKindOfClass:[CAMetalLayer class]],
			"Internal error:  layer is not CAMetalLayer");

		itsRenderer	= [[TorusGamesRenderer alloc]
						initWithLayer:	(CAMetalLayer *)theLayer
						device:			CurrentMainScreenGPU()
						multisampling:	true	//	On the one hand, in the 2D games
												//		all the antialiasing takes place
												//		in the texture filtering in the sprites' interiors
												//		(no polygon edges are visible),
												//		so there's no reason to use multisampling there.
												//	On the other hand, the 3D games will
												//		need multisampling to look their best.
						depthBuffer:	true	//	Depth buffer needed for 3D games.
						stencilBuffer:	false];

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
		//	The cursor is initially free.
		itsTorusCursorFlag = false;
#endif

		//	Respond to 2-finger trackpad gestures.
		[self setAllowedTouchTypes:NSTouchTypeMaskIndirect];	//	I'm not sure under what circumstances NSTouchTypeMaskDirect would come into play.
		itsTwoFingerGestureIsInProgress	= false;
		itsPrevGestureLocation			= CGPointZero;
	}
	return self;
}


#pragma mark -
#pragma mark lifecycle events

//	ModelData must not be locked, to avoid deadlock
//	when waiting for a possible CVDisplayLink callback to complete.
- (void)handleApplicationWillResignActiveNotification:(NSNotification *)aNotification
{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	[self exitTorusCursorMode];	//	Typically unnecessary, but safe and future-proof
#endif

	//	ModelData must not be locked, to avoid deadlock
	//	when waiting for a possible CVDisplayLink callback to complete.
	[super handleApplicationWillResignActiveNotification:aNotification];
}

- (void)handleApplicationDidBecomeActiveNotification:(NSNotification *)aNotification
{
	[super handleApplicationDidBecomeActiveNotification:aNotification];
}

//	ModelData must not be locked, to avoid deadlock
//	when waiting for a possible CVDisplayLink callback to complete.
- (void)handleWindowDidMiniaturizeNotification:(NSNotification *)aNotification
{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	[self exitTorusCursorMode];	//	Typically unnecessary, but safe and future-proof
#endif

	//	ModelData must not be locked, to avoid deadlock
	//	when waiting for a possible CVDisplayLink callback to complete.
	[super handleWindowDidMiniaturizeNotification:aNotification];
}

- (void)handleWindowDidDeminiaturizeNotification:(NSNotification *)aNotification
{
	[super handleWindowDidDeminiaturizeNotification:aNotification];
}


#pragma mark -
#pragma mark keystroke events

- (BOOL)acceptsFirstResponder
{
	ModelData	*md	= NULL;
	GameType	theGame;

	[itsModel lockModelData:&md];
	theGame = md->itsGame;
	[itsModel unlockModelData:&md];

	if (theGame == Game2DCrossword
	 && (IsCurrentLanguage(u"zs") || IsCurrentLanguage(u"zt")))
	{
		//	The Chinese Crossword game is exceptional:
		//	If we let the TorusGamesGraphicsViewMac be the first responder
		//	and accepted keystrokes directly, it would receive raw Latin letters.
		//	Instead we let the TorusGamesWindowController's itsHanziView
		//	be the first responder, receive the keystrokes,
		//	and assemble the keystrokes into completed hanzi,
		//	which the TorusGamesWindowController passes along
		//	to this TorusGamesGraphicsViewMac's -insertText: method.
		//
		//	Warning:  Even though this approach works fine,
		//	be aware that by refusing to accept first responder status,
		//	we're blocking not only keystrokes but action messages as well.
		//
		return NO;
	}
	else
	{
		//	Accept keystrokes and action messages.
		return YES;
	}
}

- (void)keyDown:(NSEvent *)anEvent
{
	switch ([anEvent keyCode])
	{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
		case ESCAPE_KEY:
			[self exitTorusCursorMode];
			break;
#endif

		case ENTER_KEY:			[self characterInput:u'\n'];	break;
		case TAB_KEY:			[self characterInput:u'\t'];	break;
		case DELETE_KEY:		[self characterInput:u'\b'];	break;
		
		case LEFT_ARROW_KEY:	[self characterInput:u'←'];		break;
		case RIGHT_ARROW_KEY:	[self characterInput:u'→'];		break;
		case DOWN_ARROW_KEY:	[self characterInput:u'↓'];		break;
		case UP_ARROW_KEY:		[self characterInput:u'↑'];		break;

		default:
			//	Process the keystroke normally.
			[self interpretKeyEvents:@[anEvent]];
			break;
	}
}

- (void)insertText:(id)aString
{
	NSString	*theNewText;
	unichar		theCharacter;
	ModelData	*md	= NULL;
	GameType	theGame;

	//	The documentation promises only that aString will be 
	//	either an NSString or an NSAttributedString. 
	//	In practice, while processing keystrokes, it's an NSString.
	if ([[aString class] isSubclassOfClass:[NSAttributedString class]])
		theNewText = [aString string];
	else
	if ([[aString class] isSubclassOfClass:[NSString class]])
		theNewText = aString;
	else
		theNewText = @"?";

	if ([theNewText length] > 0)
	{
		theCharacter = [theNewText characterAtIndex:0];

		[itsModel lockModelData:&md];
		theGame = md->itsGame;
		[itsModel unlockModelData:&md];

		if (theGame == Game2DCrossword)
		{
			//	Game2DCrossword will want character input.
			[self characterInput:theCharacter];
		}
		else
		{
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
			//	Otherwise assume the user is flailing.
			//	Perhaps s/he doesn't know how to exit torus cursor mode?
			if (itsTorusCursorFlag)
				[self errorWithTitle:	GetLocalizedText(u"TorusExitTitle")
							message:	GetLocalizedText(u"TorusExitMessage")];
#endif
		}
	}
}

- (void)characterInput:(unichar)aCharacter
{
	ModelData			*md					= NULL;
	TitledErrorMessage	*theErrorMessage	= NULL;

	[itsModel lockModelData:&md];
	theErrorMessage = CharacterInput(md, aCharacter);
	[self refreshRendererTexturesForCharacterInputWithModelData:md];
	[itsModel unlockModelData:&md];
	
	if (theErrorMessage != NULL)
	{
		[self	errorWithTitle:	theErrorMessage->itsTitle
				message:		theErrorMessage->itsMessage];
	}
}


#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE

#pragma mark -
#pragma mark torus cursor

- (void)enterTorusCursorMode
{
	ModelData	*md	= NULL;

	//	Hide the cursor and disassociate it from mouse motion
	//	so the (invisible) cursor stays parked over our window.
	//	Mouse motions will nevertheless arrive in mouseDragged calls.

	if ( ! itsTorusCursorFlag )
	{
		CGDisplayHideCursor(kCGDirectMainDisplay);	//	parameter is ignored
		CGAssociateMouseAndMouseCursorPosition(NO);
		[[self window] setAcceptsMouseMovedEvents:YES];
		itsTorusCursorFlag = true;

		[itsModel lockModelData:&md];
		md->itsChangeCount++;
		[itsModel unlockModelData:&md];
	}
}

- (void)exitTorusCursorMode
{
	ModelData	*md	= NULL;

	//	Restore the usual cursor.

	if (itsTorusCursorFlag)
	{
		CGDisplayShowCursor(kCGDirectMainDisplay);	//	parameter is ignored
		CGAssociateMouseAndMouseCursorPosition(YES);
		[[self window] setAcceptsMouseMovedEvents:NO];
		itsTorusCursorFlag = false;

		[itsModel lockModelData:&md];
		MouseGone(md);
		[itsModel unlockModelData:&md];
	}
}

#endif


#pragma mark -
#pragma mark renderer refresh

- (void)refreshRendererTexturesWithModelData:(ModelData *)md
{
	if ([itsRenderer isKindOfClass:[TorusGamesRenderer class]])	//	should never fail
		[((TorusGamesRenderer *)itsRenderer) refreshTexturesWithModelData:md];
}

- (void)refreshRendererTexturesForGameResetWithModelData:(ModelData *)md
{
	if ([itsRenderer isKindOfClass:[TorusGamesRenderer class]])	//	should never fail
		[((TorusGamesRenderer *)itsRenderer) refreshTexturesForGameResetWithModelData:md];
}

- (void)refreshRendererTexturesForCharacterInputWithModelData:(ModelData *)md
{
	if ([itsRenderer isKindOfClass:[TorusGamesRenderer class]])	//	should never fail
		[((TorusGamesRenderer *)itsRenderer) refreshTexturesForCharacterInputWithModelData:md];
}


#pragma mark -
#pragma mark mouse events

- (void)mouseMoved:(NSEvent *)anEvent
{
	[self mouseDragged:anEvent];
}

- (void)mouseDown:(NSEvent *)anEvent
{
	//	Not all Mac's have 2-button mice, so treat
	//	a control-click as a right-click.  
	//	Right clicks are important in the Apples game
	//	for marking wormy apples.
	[self mouseDown:anEvent withRightButton:
		(([anEvent modifierFlags] & NSEventModifierFlagControl) != 0)];
}

- (void)rightMouseDown:(NSEvent *)anEvent
{
	[self mouseDown:anEvent withRightButton:true];
}

- (void)mouseDown:(NSEvent *)anEvent withRightButton:(bool)aRightClick
{
	ModelData		*md				= NULL;
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	bool			theGameIs3D;
#endif
	DisplayPoint	theMouseLocation;

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE

	[itsModel lockModelData:&md];
	theGameIs3D = GameIs3D(md->itsGame);
	[itsModel unlockModelData:&md];

	if ( ! theGameIs3D && ! itsTorusCursorFlag )
		[self enterTorusCursorMode];

#endif

	theMouseLocation = [self mouseLocation:anEvent];

	//	If MouseDown() wants to pass a message to the user,
	//	it'll write it to md->itsDragMessage 
	//	for eventual display from within -mouseUpAt.

	[itsModel lockModelData:&md];
	MouseDown(	md,
				(theMouseLocation.itsX / theMouseLocation.itsViewWidth ) - 0.5,
				(theMouseLocation.itsY / theMouseLocation.itsViewHeight) - 0.5,
				[anEvent timestamp],
				[anEvent modifierFlags] & NSEventModifierFlagShift,	//	aScrollFlag
				aRightClick,										//	aMarkFlag
				[anEvent clickCount] > 1,							//	aTemporalDoubleClick
				false,												//	aTwoFingerGestureFlag
				false);												//	aFlickGestureFlag (used on iOS only)
	[itsModel unlockModelData:&md];
}

- (void)mouseDragged:(NSEvent *)anEvent
{
	ModelData			*md				= NULL;
	bool				theGameIs3D;
	DisplayPointMotion	theMouseMotion;

	[itsModel lockModelData:&md];
	theGameIs3D = GameIs3D(md->itsGame);
	[itsModel unlockModelData:&md];

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	if (itsTorusCursorFlag
	 || theGameIs3D)
#endif
	{
		theMouseMotion = [self mouseDisplacement:anEvent];

		[itsModel lockModelData:&md];
		MouseMove(	md,
					theMouseMotion.itsDeltaX / theMouseMotion.itsViewWidth,
					theMouseMotion.itsDeltaY / theMouseMotion.itsViewHeight,
					[anEvent timestamp],
					false,	//	aTwoFingerGestureFlag
					false);	//	aFlickGestureFlag
		[itsModel unlockModelData:&md];
	}
}

- (void)rightMouseDragged:(NSEvent *)anEvent
{
	[self mouseDragged:anEvent];
}

- (void)mouseUp:(NSEvent *)anEvent
{
	ModelData			*md				= NULL;
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	bool				theDoubleClickFlag;
#endif
	TitledErrorMessage	theDragMessage;

	[itsModel lockModelData:&md];
	MouseUp(md,
			[anEvent timestamp],
			false,	//	aTwoFingerGestureFlag
			false,	//	aFlickGestureFlag
			false);	//	aTouchSequenceWasCancelled
#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	theDoubleClickFlag	= md->its2DDoubleClickFlag;
#endif
	theDragMessage		= md->itsDragMessage;
	md->itsDragMessage	= (TitledErrorMessage) {NULL, NULL};
	[itsModel unlockModelData:&md];

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	if (theDoubleClickFlag)
		[self exitTorusCursorMode];
#endif
	
	//	Did MouseDown() ask us to display a message?
	if (theDragMessage.itsMessage != NULL
	 || theDragMessage.itsTitle   != NULL)
	{
		[self	errorWithTitle:	theDragMessage.itsTitle
				message:		theDragMessage.itsMessage];
	}
}

- (void)rightMouseUp:(NSEvent *)anEvent;
{
	[self mouseUp:anEvent];
}


#pragma mark -
#pragma mark trackpad events

- (void)touchesBeganWithEvent:(NSEvent *)anEvent
{
	NSSet			*theTouches;
	unsigned int	theNumTouches;
	ModelData		*md				= NULL;

	theTouches				= [anEvent touchesMatchingPhase:NSTouchPhaseAny inView:self];
	theNumTouches			= (unsigned int)[theTouches count];
	itsPrevGestureLocation	= [self gestureLocation:theTouches];
	
	//	Did a two-finger gesture begin?
	if (theNumTouches == 2)
	{
		[itsModel lockModelData:&md];

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
		//	Trackpad gestures are a bit dicey
		//	as explained in "Trackpad behavior" above.
		//	For that reason, and also to avoid issues
		//	of entering and leaving torus cursor mode,
		//	let's accept trackpad gestures only for 3D games,
		//	where they are somewhat safer and also more useful.
		//
		//	2013-12-06  Even in 2D games, two-finger scrolling
		//	is too useful to miss out on.  For simplicity,
		//	accept a two-finger drag only when the mouse
		//	is already in torus cursor mode.
		//
		//	2017-03-04  Added check that itsMouseIsDown is false,
		//	to allow hand-over-hand dragging using two fingers
		//	in alternation.
		//
		//		2017-04-04  Removed check on itsMouseIsDown,
		//		because I'm removing itsMouseIsDown altogether,
		//		and replaced it with a check that
		//		md->its2DHandStatus == HandFree .
		//
		//	2017-04-07  Using aTwoFingerGestureFlag to request
		//	scrolls keeps two-finger gesture processing
		//	(mostly? or even entirely?) separate from ordinary
		//	mouse motion processing.
		//
		//	Note:  While the (virtual) mouse button is down,
		//	the UIKit reports only one finger in motion on the trackpad.
		//	If more fingers are present, the UIKit inserts
		//	fake touchesEnded and touchesBegin events to make it
		//	look like only one finger is down at a time.
		//
		if
		(
			GameIs3D(md->itsGame)	//	3D games always accept two-finger gestures.
		 ||
			(
				itsTorusCursorFlag	//	2D games accept two-finger gestures only in torus cursor mode.
			 &&
				md->its2DHandStatus == HandFree
			)
		)
#endif
		{
			MouseDown(	md,
						itsPrevGestureLocation.x,
						itsPrevGestureLocation.y,
						[anEvent timestamp],
						false,	//	aScrollFlag
						false,	//	aMarkFlag
						false,	//	aTemporalDoubleClick
						true,	//	aTwoFingerGestureFlag
						false);	//	aFlickGestureFlag (used on iOS only)

			itsTwoFingerGestureIsInProgress = true;
		}

		[itsModel unlockModelData:&md];
	}
	else
	{
		//	If the user touches a third finger to the trackpad,
		//	abort the two-finger gesture.
		if (itsTwoFingerGestureIsInProgress)
		{
			[itsModel lockModelData:&md];
			MouseUp(md,
					[anEvent timestamp],
					true,	//	aTwoFingerGestureFlag
					false,	//	aFlickGestureFlag
					false);	//	aTouchSequenceWasCancelled
			[itsModel unlockModelData:&md];

			itsTwoFingerGestureIsInProgress = false;
		}
	}
}

- (void)touchesMovedWithEvent:(NSEvent *)anEvent
{
	NSSet		*theTouches;
	CGPoint		theGestureLocation;	//	in [-0.5, +0.5] trackpad coordinates
	ModelData	*md	= NULL;

	//	When the user lifts a finger from the touch pad,
	//	we may get one last touchesMoved event, in which
	//	the lifted finger's touch is still present but
	//	with phase NSTouchPhaseEnded, before we get
	//	the touchesEnded event announcing that touch's
	//	complete removal.  In order to include the recently
	//	lifted finger's touch in theTouches, ask for touches
	//	matching NSTouchPhaseAny rather than NSTouchPhaseTouching.
	//
	theTouches			= [anEvent touchesMatchingPhase:NSTouchPhaseAny inView:self];
	theGestureLocation	= [self gestureLocation:theTouches];
	
	//	Is a two-finger gesture in progress?
	if (itsTwoFingerGestureIsInProgress)
	{
		[itsModel lockModelData:&md];
		MouseMove(	md,
					theGestureLocation.x - itsPrevGestureLocation.x,
					theGestureLocation.y - itsPrevGestureLocation.y,
					[anEvent timestamp],
					true,	//	aTwoFingerGestureFlag
					false);	//	aFlickGestureFlag
		[itsModel unlockModelData:&md];
		
		itsPrevGestureLocation = theGestureLocation;
	}
}

- (void)touchesEndedWithEvent:(NSEvent *)anEvent
{
	ModelData	*md	= NULL;

	//	Is a two-finger gesture ending?
	if (itsTwoFingerGestureIsInProgress)
	{
		[itsModel lockModelData:&md];
		MouseUp(md,
				[anEvent timestamp],
				true,	//	aTwoFingerGestureFlag
				false,	//	aFlickGestureFlag
				false);	//	aTouchSequenceWasCancelled
		[itsModel unlockModelData:&md];
			
		itsTwoFingerGestureIsInProgress	= false;
		itsPrevGestureLocation			= CGPointZero;
	}
}

- (void)touchesCancelledWithEvent:(NSEvent *)anEvent
{
	//	Apple documentation at
	//
	//		https://developer.apple.com/library/content/documentation/Cocoa/Conceptual/EventOverview/HandlingTouchEvents/HandlingTouchEvents.html
	//
	//	says that
	//
	//		The operating system calls touchesCancelledWithEvent: when an external event
	//		— for example, application deactivation — interrupts the current multitouch sequence.
	//
	//	so presumably this will get called rarely if ever.
	
	[self touchesEndedWithEvent:anEvent];
}

- (CGPoint)gestureLocation:(NSSet *)aTouchSet	//	in [-0.5, +0.5] trackpad coordinates
{
	CGPoint			theGestureLocation;
	NSTouch			*theTouch;
	CGPoint			theTouchPoint;
	unsigned int	theNumTouches;
	
	theGestureLocation.x = 0.0;
	theGestureLocation.y = 0.0;

	theNumTouches = 0;

	for (theTouch in aTouchSet)
	{
		//	Get theTouchPoint in [0,1] × [0,1] trackpad coordinates.
		theTouchPoint = [theTouch normalizedPosition];
		
		theGestureLocation.x += theTouchPoint.x;
		theGestureLocation.y += theTouchPoint.y;
		
		theNumTouches++;
	}
	
	if (theNumTouches > 0)
	{
		theGestureLocation.x /= theNumTouches;
		theGestureLocation.y /= theNumTouches;
	}
		
	//	Convert theGestureLocation from [0,1] × [0,1] coordinates
	//	to [-0.5, +0.5] × [-0.5, +0.5] coordinates.
	theGestureLocation.x -= 0.5;
	theGestureLocation.y -= 0.5;
	
	return theGestureLocation;
}


#pragma mark -
#pragma mark error message

- (void)errorWithTitle:(ErrorText)aTitle message:(ErrorText)aMessage
{
	//	Call this method only with the ModelData unlocked.

#ifdef TORUS_GAMES_2D_MOUSE_INTERFACE
	if (itsTorusCursorFlag)
		[self exitTorusCursorMode];
#endif

	GeometryGamesErrorMessage(aMessage, aTitle);
}


@end
